<?php
/**
 * Multi Blog Hub Child Theme Functions
 *
 * All child theme functions consolidated into single file.
 * Order matters: pluggable function overrides must come first.
 *
 * @package Multi_Blog_Hub
 * @since 1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/*
--------------------------------------------------------------
PLUGGABLE FUNCTION OVERRIDES. (Must be defined before parent loads.)
--------------------------------------------------------------
*/

/**
 * Set up custom header with child theme default image.
 *
 * @since 1.0.0
 */
function good_looking_blog_custom_header_setup() { // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedFunctionFound -- Overrides parent theme pluggable function.
	add_theme_support(
		'custom-header',
		apply_filters(
			'good_looking_blog_custom_header_args', // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Must match parent theme filter name.
			array(
				'default-image'      => get_stylesheet_directory_uri() . '/images/header-image.jpg',
				'default-text-color' => '#000000',
				'width'              => 1920,
				'height'             => 760,
				'flex-height'        => true,
				'video'              => true,
				'wp-head-callback'   => 'good_looking_blog_header_style',
			)
		)
	);

	register_default_headers(
		array(
			'default-image' => array(
				'url'           => get_stylesheet_directory_uri() . '/images/header-image.jpg',
				'thumbnail_url' => get_stylesheet_directory_uri() . '/images/header-image.jpg',
				'description'   => __( 'Default Header Image', 'multi-blog-hub' ),
			),
		)
	);
}

/**
 * Get custom font face styles for Manrope font.
 *
 * @since 1.0.0
 * @return string Font face CSS rules.
 */
function good_looking_blog_get_font_face_styles() { // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedFunctionFound -- Overrides parent theme pluggable function.

	return "
        @font-face{
            font-family: 'Manrope';
            font-weight: 400;
            font-style: normal;
            font-stretch: normal;
            font-display: swap;
            src: url('" . esc_url( get_theme_file_uri( 'fonts/Manrope-Regular.ttf' ) ) . "') format('truetype');
        }

        @font-face{
            font-family: 'Manrope';
            font-weight: 600;
            font-style: normal;
            font-stretch: normal;
            font-display: swap;
            src: url('" . esc_url( get_theme_file_uri( 'fonts/Manrope-SemiBold.ttf' ) ) . "') format('truetype');
        }

        @font-face{
            font-family: 'Manrope';
            font-weight: 700;
            font-style: normal;
            font-stretch: normal;
            font-display: swap;
            src: url('" . esc_url( get_theme_file_uri( 'fonts/Manrope-Bold.ttf' ) ) . "') format('truetype');
        }
        ";
}

/**
 * Filter footer sidebars to use 3 columns instead of 4.
 *
 * @since 1.0.0
 * @return array Modified sidebar IDs.
 */
function multi_blog_hub_filter_footer_sidebars() {
	return array( 'footer-one', 'footer-two', 'footer-three' );
}
add_filter( 'good_looking_blog_footer_sidebars', 'multi_blog_hub_filter_footer_sidebars' );

/**
 * Display visible header search form (respects Enable Search toggle).
 *
 * @since 1.0.0
 */
function good_looking_blog_header_search() { // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedFunctionFound -- Overrides parent theme pluggable function.
	$ed_search = get_theme_mod( 'ed_search', true );

	// If search is disabled in Customizer, do not output anything.
	if ( ! $ed_search ) {
		return;
	}
	?>
	<div class="header-search">
		<form role="search" method="get" class="search-form" action="<?php echo esc_url( home_url( '/' ) ); ?>">
			<label class="screen-reader-text" for="header-search-field">
				<?php esc_html_e( 'Search for:', 'multi-blog-hub' ); ?>
			</label>
			<div class="search-input-wrapper">
				<input 
					type="search"
					id="header-search-field"
					class="search-field"
					placeholder="<?php esc_attr_e( 'Search an article....', 'multi-blog-hub' ); ?>"
					value="<?php echo esc_attr( get_search_query() ); ?>"
					name="s"
				/>
				<button type="submit" class="search-submit" aria-label="<?php esc_attr_e( 'Search', 'multi-blog-hub' ); ?>">
					<svg width="19" height="19" viewBox="0 0 19 19" fill="none" xmlns="http://www.w3.org/2000/svg">
						<path d="M18.5002 18.5002L14.1572 14.1572M14.1572 14.1572C14.9001 13.4143 15.4894 12.5324 15.8914 11.5618C16.2935 10.5911 16.5004 9.55081 16.5004 8.50021C16.5004 7.4496 16.2935 6.40929 15.8914 5.43866C15.4894 4.46803 14.9001 3.58609 14.1572 2.84321C13.4143 2.10032 12.5324 1.51103 11.5618 1.10898C10.5911 0.706931 9.55081 0.5 8.50021 0.5C7.4496 0.5 6.40929 0.706931 5.43866 1.10898C4.46803 1.51103 3.58609 2.10032 2.84321 2.84321C1.34288 4.34354 0.5 6.37842 0.5 8.50021C0.5 10.622 1.34288 12.6569 2.84321 14.1572C4.34354 15.6575 6.37842 16.5004 8.50021 16.5004C10.622 16.5004 12.6569 15.6575 14.1572 14.1572Z" stroke="#6B6B6B" stroke-linecap="round" stroke-linejoin="round"/>
					</svg>
				</button>
			</div>
		</form>
	</div><!-- .header-search -->
	<?php
}

/*
--------------------------------------------------------------
CHILD THEME FUNCTIONS. (Hooked functions with child prefix.)
--------------------------------------------------------------
*/

/**
 * Enqueue parent and child theme stylesheets.
 *
 * @since 1.0.0
 */
function multi_blog_hub_styles() {
	$my_theme = wp_get_theme();
	$version  = $my_theme->get( 'Version' );

	// Enqueue the parent theme's stylesheet.
	wp_enqueue_style( 'good-looking-blog', get_template_directory_uri() . '/style.css', array(), $version );

	// Add font face styles inline.
	wp_add_inline_style( 'multi-blog-hub-font-style', good_looking_blog_get_font_face_styles() );
}
add_action( 'wp_enqueue_scripts', 'multi_blog_hub_styles' );

/*
--------------------------------------------------------------
// FILTERS.
--------------------------------------------------------------
*/

/**
 * Customize homepage sections.
 *
 * @since 1.0.0
 * @return array Modified sections array.
 */
function multi_blog_hub_home_sections() {
	return array( 'banner', 'blog', 'editor-pick', 'newsletter', 'instagram' );
}
add_filter( 'good_looking_blog_home_sections', 'multi_blog_hub_home_sections' );

/**
 * Display post date without link.
 *
 * Overrides parent theme's posted_on function to remove link from date.
 *
 * @since 1.0.0
 */
function good_looking_blog_posted_on() { // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedFunctionFound -- Overrides parent theme pluggable function.
	$time_string = '<time class="entry-date published updated" datetime="%1$s" itemprop="datePublished">%2$s</time><time class="updated" datetime="%3$s" itemprop="dateModified">%4$s</time>';

	$time_string = sprintf(
		$time_string,
		esc_attr( get_the_date( 'c' ) ),
		esc_html( get_the_date() ),
		esc_attr( get_the_modified_date( 'c' ) ),
		esc_html( get_the_modified_date() )
	);

	echo '<span class="posted-on">' . $time_string . '</span>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- $time_string is already escaped.
}

/**
 * Display footer copyright with date.
 *
 * Overrides parent theme's footer copyright function to fix date display bug.
 *
 * @since 1.0.0
 */
function good_looking_blog_get_footer_copyright() { // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedFunctionFound -- Overrides parent theme pluggable function.
	$copyright = get_theme_mod( 'footer_copyright' );
	echo '<span class="copy-right">';
	if ( ! empty( $copyright ) ) {
		echo wp_kses_post( $copyright );
	} else {
		echo esc_html__( 'Copyright &copy; ', 'multi-blog-hub' ) . esc_html( date_i18n( 'Y' ) );
		echo '<a href="' . esc_url( home_url( '/' ) ) . '">' . esc_html( get_bloginfo( 'name' ) ) . '</a>. ';
	}
	echo '</span>';
}

/**
 * Unregister parent theme's Homepage Advertisement widget area.
 *
 * @since 1.0.1
 */
function multi_blog_hub_unregister_home_ad_sidebar() {
	unregister_sidebar( 'ad-home' );
}
add_action( 'widgets_init', 'multi_blog_hub_unregister_home_ad_sidebar', 20 );
